// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * Class ZoomCanvas manages the conversion of mouse
 * gestures into positioning the zoom reticle on the
 * screen and to draw the reticle.
 */

package hairSynth;

import processing.core.PApplet;

final class ZoomCanvas {

	// All vars private
	private boolean isZoomCtrSetByUser = false;
	private boolean isRedrawNeeded = true;
	private boolean isValidDrag = false;

	private static final float halfSidePix3x = K.getZoomBoxHalfSide(K.ixViewZoom3x) * K.canvasScale;
	private static final float halfSidePix10x = K.getZoomBoxHalfSide(K.ixViewZoo10x) * K.canvasScale;
	// private static final float halfSidePix30x = K.getZoomBoxHalfSide(K.ixViewZoo30x) * K.CANVAS_SCALE;
	private static final float halfSidePix100x = K.getZoomBoxHalfSide(K.ixViewZo100x) * K.canvasScale;

	private float zoomCtrXpix; // relative to canvas! not to applet.
	private float zoomCtrYpix; // relative to canvas! not to applet.
	private boolean isHover; // meaning ready to drag
	private boolean isHoverAsOfLastDraw;

	private float lastDragDX;
	private float lastDragDY;

	private float pressCanvasX;
	private float pressCanvasY;

	private float zoomCtrXonPress;
	private float zoomCtrYonPress;

	private float rgbRed = 0.0f; // inverse of "reddish"
	private float rgbGrn = 192.0f;
	private float rgbBlu = 192.0f;

	private PApplet papplet;

	ZoomCanvas(final PApplet papplet_) {
		this.papplet = papplet_;

		this.setZoomCtrPixPosition(0.5f * K.canvasScale, 0.5f * K.canvasScale);
		this.isHover = false;
	}

	void triggerIsRedrawNeeded() {
		this.isRedrawNeeded = true;
	}

	void setIsZoomCtrSetByUser(final boolean bool) {
		this.isZoomCtrSetByUser = bool;
	}

	private void testHoverOverReticle(final float mouseX, final float mouseY) {

		float canvasX = mouseX - K.canvas_x0;
		if ((canvasX >= (this.zoomCtrXpix - halfSidePix10x)) && (canvasX <= (this.zoomCtrXpix + halfSidePix10x))
				&& (mouseY >= (this.zoomCtrYpix - halfSidePix10x)) && (mouseY <= (this.zoomCtrYpix + halfSidePix10x))) {
			this.isHover = true;
		} else {
			this.isHover = false;
		}
		if (this.isHover != this.isHoverAsOfLastDraw) {
			this.isRedrawNeeded = true; // changed color
		}
	}

	boolean recordZoomCenterPositionOnPressIfValidDrag(final float mouseX, final float mouseY) {

		this.lastDragDX = 0.0f;
		this.lastDragDY = 0.0f;

		testHoverOverReticle(mouseX, mouseY);
		this.isValidDrag = this.isHover;
		if (this.isValidDrag) {

			this.pressCanvasX = mouseX - K.canvas_x0;
			this.pressCanvasY = mouseY;

			this.zoomCtrXonPress = this.zoomCtrXpix;
			this.zoomCtrYonPress = this.zoomCtrYpix;
			return true;
		}
		return false;
	}

	boolean moveZoomCenterPositionOnDrag(final float mouseX, final float mouseY) {

		float dragDX = mouseX - K.canvas_x0 - this.pressCanvasX;
		float dragDY = mouseY - this.pressCanvasY;

		if (this.isValidDrag && ((this.lastDragDX != dragDX) || (this.lastDragDY != dragDY))) { // was dragged

			setZoomCtrPixPosition(this.zoomCtrXonPress + dragDX, this.zoomCtrYonPress + dragDY);

			this.lastDragDX = dragDX;
			this.lastDragDY = dragDY;

			this.isZoomCtrSetByUser = true;
			this.isRedrawNeeded = true; // changed position
			return true;
		}
		return false;
	}

	void setZoomCtrPixPosition(final float canvasX_, final float canvasY_) {

		this.zoomCtrXpix = Math.max(0, Math.min(K.canvasScale, canvasX_));
		this.zoomCtrYpix = Math.max(0, Math.min(K.canvasScale, canvasY_));

		K.setCtrZoomXandY(this.zoomCtrXpix, this.zoomCtrYpix);
	}

	boolean drawIfNeeded(final BufferBunch bufferBunch_, final float mouseX, final float mouseY) {

		this.testHoverOverReticle(mouseX, mouseY); // update hover ... may trigger reDraw
		if (bufferBunch_.getAndResetIsCopyToScreenSinceLastPoll() || this.isRedrawNeeded) {

			// ----- update reticle position if not already set by user
			if (!this.isZoomCtrSetByUser) {
				this.setZoomCtrPixPosition(bufferBunch_.getZoomAvgYpix(), bufferBunch_.getZoomAvgXpix());
			}

			if (this.isHover) { // keep green unchanged for shift in color
				this.papplet.stroke(this.rgbRed, this.rgbGrn, this.rgbBlu);
			} else {
				this.papplet.stroke(Math.max(0.0f, this.rgbRed - 96.0f), Math.max(0.0f, this.rgbGrn - 96.0f), Math.max(
						0.0f, this.rgbBlu - 96.0f));
			}
			this.papplet.noFill();
			drawClippedReticle(halfSidePix3x);
			drawClippedReticle(halfSidePix10x);
			drawClippedReticle(halfSidePix10x - 1.0f);
			drawClippedReticle(halfSidePix10x - 2.0f);
			drawClippedReticle(halfSidePix100x);

			if (this.isHover) {
				this.papplet.fill(this.rgbRed, this.rgbGrn, this.rgbBlu);
			} else {
				this.papplet.fill(Math.max(0.0f, this.rgbRed - 96.0f), Math.max(0.0f, this.rgbGrn - 96.0f), Math.max(
						0.0f, this.rgbBlu - 96.0f));
			}

			this.papplet.text("3x", this.zoomCtrXpix + 116 + K.canvas_x0, this.zoomCtrYpix - 3 - halfSidePix3x);
			this.papplet.text("10x", this.zoomCtrXpix + 14 + K.canvas_x0, this.zoomCtrYpix - 3 - halfSidePix10x);
			if (this.zoomCtrXpix > 14) {
				this.papplet.text("100x", this.zoomCtrXpix - 14 + K.canvas_x0, this.zoomCtrYpix - 7);
			}

			this.isHoverAsOfLastDraw = this.isHover;
			this.isRedrawNeeded = false; // resets since just drawn
			return true;
		}
		return false;
	}

	private void drawClippedReticle(final float halfSide) {

		float x0 = this.zoomCtrXpix - halfSide + K.canvas_x0;
		if (x0 >= K.canvas_x0) {
			this.papplet.rect(x0, this.zoomCtrYpix - halfSide, 2.0f * halfSide, 2.0f * halfSide);
		} else {
			float x1 = this.zoomCtrXpix + halfSide + K.canvas_x0;
			float y0 = this.zoomCtrYpix - halfSide;
			float y1 = this.zoomCtrYpix + halfSide;
			this.papplet.line(K.canvas_x0, y0, x1, y0);
			this.papplet.line(K.canvas_x0, y1, x1, y1);
			this.papplet.line(x1, y0, x1, y1);
		}
	}

	void setGraticuleColorToCurrentScheme() {

		if (KColor.ixColrButOn == KColor.cat5_COLOR) { // if cat5 ... use Magenta

			this.rgbRed = 255.0f;
			this.rgbGrn = 0.0f;
			this.rgbBlu = 255.0f;

		} else {

			int rgb = KColor.getInverseOfCurrColorRGB();
			this.rgbRed = (rgb >> 16) & 0xFF;
			this.rgbGrn = (rgb >> 8) & 0xFF;
			this.rgbBlu = rgb & 0xFF;
		}
	}
}
