// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * Class Strand holds the information needed to draw
 * each successive segment of the hair strands.
 */

package hairSynth;

final class Strand {

	// No public vars here
	private static double maxDegreesChangePerSegment = 2.0; // express in degrees ... changed to 2
	private static double maxRadiansChangePerSegment = maxDegreesChangePerSegment * K.toRADIANS;

	private int myId;
	private int myBinary;

	private double face0X;
	private double face0Y;
	private double face0N;

	private double face1X;
	private double face1Y;
	private double face1N;

	private double face1DxInit;
	private double face1DyInit;
	private double face1nInit;

	private Oscill3 osc3;

	private float deltaHue; // change in Hue .. to be added to basic KColor.hue[][]
	private float deltaSat;
	private float deltaBri;

	private int cat5color;

	Strand(final int myId_) {
		this.myId = myId_;
		this.myBinary = 0;
		this.osc3 = new Oscill3();
		this.clearDeltaHSBandCat5color();
	}

	int getMyBinary() {
		return this.myBinary;
	}

	double getFace1DXInit() {
		return K.face1xINIT + this.face1DxInit;
	}

	double getFace1DYInit() {
		return K.face1yINIT + this.face1DyInit;
	}

	double getFace1nInit() {
		return this.face1nInit;
	}

	void setPositionDxDyInit() {
		this.face1DxInit = 0.0;
		this.face1DyInit = 0.0;
	}

	void setFace1nInit(final double normalDir) {
		this.face1nInit = normalDir;
	}

	void setOscParams(final int ixOsc, final double period_, final double mult0_, final double mult1_) {
		this.osc3.setOscParams(ixOsc, period_, mult0_, mult1_);
	}

	void copyParamsFrom(final Strand sourceStrand_) {

		this.face1DxInit = sourceStrand_.getFace1DXInit();
		this.face1DyInit = sourceStrand_.getFace1DYInit();
		this.face1nInit = sourceStrand_.getFace1nInit();

		this.osc3.copyOscParamsFrom(sourceStrand_.osc3);
	}

	void copyAndJiggleParams(final Strand sourceStrand_, final double jiggleFactor) {

		double stepDX = K.radiusXtuftROOT;
		double stepDY = K.radiusYtuftROOT;
		double dXaccum = stepDX;
		double dYaccum = -stepDY;
		int power = Buffer.numberOfStrandsTRAVERSED / 2;

		for (int exp = Buffer.maxLEVEL; exp >= 0; exp--) {

			if ((exp % 2) == 0) { // deal with Y
				if ((this.myId & power) > 0) {
					dYaccum += stepDY;
				}
				stepDY /= 2.0f; // goes down once in two cycles
			} else { // deal with X
				if ((this.myId & power) > 0) {
					dXaccum -= stepDX;
				}
				stepDX /= 2.0f; // goes down once in two cycles
			}
			power /= 2; // goes down once by cycle
		}

		double absDXaccum = Math.abs(dXaccum);
		double absDYaccum = Math.abs(dYaccum);
		double radius = Math.max(absDXaccum, absDYaccum);
		if (radius > 0.0f) {
			double hypo = Math.sqrt(dXaccum * dXaccum + dYaccum * dYaccum);
			double shrink = radius / hypo; // always positive number
			dXaccum *= shrink;
			dYaccum *= shrink;
		}

		this.face1DxInit = dXaccum + 2.0 * stepDX * (Math.random() - 0.5);
		this.face1DyInit = dYaccum + 2.0 * stepDY * (Math.random() - 0.5);

		this.face1nInit = sourceStrand_.getFace1nInit();

		this.osc3.copyAndJiggleOscParams(sourceStrand_.osc3, jiggleFactor);
	}

	float getDeltaHue() {
		return this.deltaHue;
	}

	float getDeltaSat() {
		return this.deltaSat;
	}

	float getDeltaBri() {
		return this.deltaBri;
	}

	void setDeltaHSB(final float deltaHue_, final float deltaSat_, final float deltaBri_) {
		this.deltaHue = deltaHue_;
		this.deltaSat = deltaSat_;
		this.deltaBri = deltaBri_;
	}

	void clearDeltaHSBandCat5color() {
		this.deltaHue = 0.0f;
		this.deltaSat = 0.0f;
		this.deltaBri = 0.0f;
		this.cat5color = 0;
	}

	int getCat5color() {
		return this.cat5color;
	}

	void setCat5color(final int ixColor) {
		this.cat5color = ixColor;
	}

	// ----- these used by DRAW .. converted to float
	float getFace0Xfloat() {
		return (float) this.face0X;
	}

	float getFace0Yfloat() {
		return (float) this.face0Y;
	}

	float getFace0NormalFloat() {
		return (float) this.face0N;
	}

	float getFace1Xfloat() {
		return (float) this.face1X;
	}

	float getFace1Yfloat() {
		return (float) this.face1Y;
	}

	float getFace1NormalFloat() {
		return (float) this.face1N;
	}

	double nextSegment(final double dist) {

		double radius = this.osc3.calcRadius(dist);

		// ----- calculate segment length: maximum of change in orientation; remaining to next phase change or strand
		double maxSegmentLen = Math.abs(maxRadiansChangePerSegment * radius); // limited by segment change of
		// orientation
		maxSegmentLen = Math.min(maxSegmentLen, K.maxSegmentLENGTH[K.ixZoomSwatchButOn]); // avoid large segments
		maxSegmentLen = Math.min(maxSegmentLen, K.totalStrandLEN - dist); // limited by length of strand

		double distBefReversion = this.osc3.calcDistBefFirstReversion(dist);
		if (maxSegmentLen > distBefReversion) {
			maxSegmentLen = distBefReversion + K.minSegmentLENGTH; // to force going to next phase
		} else {
			maxSegmentLen = Math.max(maxSegmentLen, K.minSegmentLENGTH); // to ensure minimum advance
		}

		if (dist == 0.0) { // initialize faces
			this.face1X = this.face1DxInit + K.face1xINIT;
			this.face1Y = this.face1DyInit + K.face1yINIT;
			this.face1N = this.face1nInit;
		}

		// ----- push face1 into face0
		this.face0X = this.face1X;
		this.face0Y = this.face1Y;
		this.face0N = this.face1N;

		// ----- calculate change in orientation
		double changeInRadians = maxSegmentLen / radius;

		this.face1N += changeInRadians;

		this.face1N = K.forceFromMinusPiToPlusPi(this.face1N); // normalize

		// ----- calculate face1 X
		double dX = radius * (Math.sin(this.face1N) - Math.sin(this.face0N));
		this.face1X = this.face0X + dX;

		// ----- calculate face1 Y
		double dY = radius * (-Math.cos(this.face1N) + Math.cos(this.face0N));
		this.face1Y = this.face0Y + dY;

		return (dist + maxSegmentLen);
	}
}
