
//	 SimplePostscript v.0092
//	 Simple Java library for outputting Postscript graphics.
//	 See http://process.unlekker.net/SimplePostscript/
//
//	 Marius Watz - www.unlekker.net
// ----------------------------------------------------------------------------
//   If you plan to use this class in your own project, please do not copy this
//   file but rather download the latest version of the library at
//   http://processing.unlekker.net/SimplePostscript/index.html
// ----------------------------------------------------------------------------
//   Note: JMBara updated PConstant.RADIUS_CENTER to RADIUS for compliance with
//         latest processing PConstant library.

package hairSynth;

import java.io.FileWriter;
import java.io.IOException;

import processing.core.PApplet;

/**
 * <p>
 * Title: SimplePostscript
 * </p>
 *
 * <p>
 * Description: Library to write PostScript files from Java or <a href="http://www.processing.org/"
 * target="_blank">Processing</a>. Use by calling the open() function with a filename and coordinates for the bounding
 * box. open() returns a SimplePostscript instance which can be used to draw to the opened file.
 * </p>
 *
 * <p>
 * Copyright: Copyright (c) 2005, Marius Watz
 * </p>
 *
 * @author not attributable
 * @version 0092
 */
public class SimplePostscript implements processing.core.PConstants {
	public boolean isOpen;
	FileWriter out;
	PApplet p;

	public SimplePostscript(final PApplet _parent) {
		this.p = _parent;
	}

	/**
	 * Opens a new Postscript file. (x1,y1) - (x2,y2) are the coordinates that define the bounding box. PostScript
	 * interpreters might choose to ignore drawing outside this area.
	 *
	 * @param filename
	 *            String - name of file to be opened.
	 * @param x1
	 *            float
	 * @param y1
	 *            float
	 * @param x2
	 *            float
	 * @param y2
	 *            float
	 * @return success - True if successful, false otherwise.
	 */
	public final boolean open(final String filename_, final float x1, final float y1, final float x2, final float y2) {

		String filename = this.p.savePath(filename_);
		System.out.println("SimplePostscript.open(" + filename + ")");

		try {
			this.out = new FileWriter(filename);
			this.isOpen = true;
			println("%!PS-Adobe-3.0 EPSF-3.0");
			println("%%Creator: SimplePostscript");
			println("%%BoundingBox: " + fmt(x1) + " " + fmt(y1) + " " + fmt(x2) + " " + fmt(y2));
			println("%%HiResBoundingBox: " + fmt(x1) + " " + fmt(y1) + " " + fmt(x2) + " " + fmt(y2));
		} catch (Exception e) {
			this.isOpen = false;

			System.out.println("Exception opening file: " + filename);
			return false;
		}

		return true;
	}

	public final boolean open(final String filename) {
		return open(filename, 0, 0, this.p.width, this.p.height);
	}

	public final void setlinewidth(final float w) {
		println(fmt(w) + " setlinewidth");
	}

	public final void setlinejoin(final int c) {
		println(fmt(c) + " setlinejoin");
	}

	public final void setlinecap(final int c) {
		println(fmt(c) + " setlinecap");
	}

	public final void setdash(final String str) {
		println(str + " setdash");
	}

	public final void setgray(final float g) {
		println(fmt(g) + " setgray");
	}

	public final void setcmyk(final float c, final float m, final float y, final float k) {
		println(fmt(c) + " " + fmt(m) + " " + fmt(y) + " " + fmt(k) + " setcmykcolor");
	}

	public final void setrgb(final int r, final int g, final int b) {
		println(fmt((float) r / 255) + " " + fmt((float) g / 255) + " " + fmt((float) b / 255) + " setrgbcolor");
	}

	public final void setrgb(final float r, final float g, final float b) {
		println(fmt(r / 255) + " " + fmt(g / 255) + " " + fmt(b / 255) + " setrgbcolor");
	}

	public final void setfont(final String fontname) {
		println("/" + fontname + " findfont setfont");
	}

	public final void setfont(final String fontname, final float pt) {
		println("/" + fontname + " findfont " + fmt(pt) + " scalefont setfont");
	}

	public final void moveto(final float x, final float y) {
		println(fmt(x) + " " + fmt(this.p.height - y) + " moveto");
	}

	public final void lineto(final float x, final float y) {
		println(fmt(x) + " " + fmt(this.p.height - y) + " lineto");
	}

	public final void rect(final float x1_, final float y1_, final float x2_, final float y2_) {
		float x1 = x1_;
		float y1 = y1_;
		float x2 = x2_;
		float y2 = y2_;

		float hradius, vradius;
		switch (this.p.g.rectMode) {
		case CORNERS:
			break;
		case CORNER:
			x2 += x1;
			y2 += y1;
			break;
		case RADIUS: // JMB note: was CENTER_RADIUS
			hradius = x2;
			vradius = y2;
			x2 = x1 + hradius;
			y2 = y1 + vradius;
			x1 -= hradius;
			y1 -= vradius;
			break;
		case CENTER:
			hradius = x2 / 2.0f;
			vradius = y2 / 2.0f;
			x2 = x1 + hradius;
			y2 = y1 + vradius;
			x1 -= hradius;
			y1 -= vradius;
			break;
			default:
				/* do nothing */
				break;
		}

		if (x1 > x2) {
			float temp = x1;
			x1 = x2;
			x2 = temp;
		}

		if (y1 > y2) {
			float temp = y1;
			y1 = y2;
			y2 = temp;
		}

		moveto(x1, y1);
		lineto(x2, y1);
		lineto(x2, y2);
		lineto(x1, y2);
		lineto(x1, y1);
	}

	public final void arc(final float x, final float y, final float rad, final float deg1, final float deg2) {
		println(fmt(x) + " " + " " + fmt(this.p.height - y) + " " + fmt(rad) + " " + fmt(deg1) + " " + fmt(deg2)
				+ " arc");
	}

	public final void arcNegative(final float x, final float y, final float rad, final float deg1, final float deg2) {
		println(fmt(x) + " " + " " + fmt(this.p.height - y) + " " + fmt(rad) + " " + fmt(deg1) + " " + fmt(deg2)
				+ " arcn");
	}

	public final void circle(final float x, final float y, final float rad) {
		moveto(x + rad, y);
		arc(x, y, rad, 0, 360);
	}

	public final void ellipse(final float a, final float b, final float c, final float d) {
		float x = a, y = b, w = c, h = d;
		float kappa = 0.5522847498f;

		if (this.p.g.ellipseMode == CORNERS) {
			w = (c - a) / 2;
			h = (d - b) / 2;
			x = a + w;
			y = b + h;
		} else if (this.p.g.ellipseMode == RADIUS) { // JMB note: was CENTER_RADIUS
			x = a - c;
			y = b - d;
			w = c * 2;
			h = d * 2;

		} else if (this.p.g.ellipseMode == CORNER) {
			w /= 2;
			h /= 2;
			x = a + w;
			y = b + h;
		}

		if (w < 0) { // undo negative width
			x += w;
			w = -w;
		}

		if (h < 0) { // undo negative height
			y += h;
			h = -h;
		}

		y = this.p.height - y;

		moveto(w + x, 0 + y);
		curveto(w + x, kappa * h + y, kappa * w + x, h + y, 0 + x, h + y);
		curveto(-kappa * w + x, h + y, -w + x, kappa * h + y, -w + x, 0 + y);
		curveto(-w + x, -kappa * h + y, -kappa * w + x, -h + y, 0 + x, -h + y);
		curveto(kappa * w + x, -h + y, w + x, -kappa * h + y, w + x, 0 + y);
	}

	public final void curveto(final float x1, final float y1, final float x2, final float y2, final float x3, final float y3) {
		println("" + fmt(x1) + " " + fmt(this.p.height - y1) + " " + fmt(x2) + " " + fmt(this.p.height - y2) + " "
				+ fmt(x3) + " " + fmt(this.p.height - y3) + " curveto");
	}

	public final void fill() {
		println("fill");
	}

	public final void fillclosepath() {
		println("closepath fill");
	}

	public final void stroke() {
		println("stroke");
	}

	public final void strokeclosepath() {
		println("closepath stroke");
	}

	public final void fillstroke() {
		gsave();
		stroke();
		grestore();
		fill();
	}

	public final void fillstrokeclosepath() {
		gsave();
		strokeclosepath();
		grestore();
		fillclosepath();
	}

	public final void rotate(final float deg) {
		println(fmt(deg) + " rotate");
	}

	public final void scale(final float x, final float y) {
		println(fmt(x) + " " + fmt(y) + " scale");
	}

	public final void translate(final float x, final float y) {
		println(fmt(x) + " " + fmt(-y) + " translate");
	}

	public final void gsave() {
		println("gsave");
	}

	public final void grestore() {
		println("grestore");
	}

	public final void text(final float x, final float y, final String s) {
		if (emptyString(s)) {
			return;
		}
		moveto(x, y);
		println("(" + formatString(s) + ") show");
	}

	public final void println(final String s) {
		if (!this.isOpen) {
			System.err.println("SimplePostscript.println(" + s + "): File not open.");
			return;
		}

		try {
			this.out.write(s + "\n");
		} catch (IOException e) {
			System.out.println("IOException!");
		}
	}

	public final void print(final String s) {
		if (!this.isOpen) {
			System.err.println("SimplePostscript.print(" + s + "): File not open.");
			return;
		}
		try {
			this.out.write(s);
		} catch (IOException e) {
			System.out.println("IOException!");
		}
	}

	private static String fmt(final float f_) {
		double m = Math.pow(10, 3);
		float f = Math.round(f_ * m);
		f /= m;

		return "" + f;
	}

	private String formatString(final String str) {
		String s;
		char c;
		int i;

		s = "";
		for (i = 0; i < str.length(); i++) {
			c = str.charAt(i);
			if (c == '(') {
				s = s + "\\(";
			} else if (c == ')') {
				s = s + "\\)";
			} else if (c == '\\') {
				s = s + "\\\\";
			} else if (c != '\n'){ // do nothing if (c == '\n')
				s = s + "" + c;
			}
		}
		return s;
	}

	private boolean emptyString(final String str) {
		// java.lang.Character c;

		for (int i = 0; i < str.length(); i++) {
			if (! Character.isSpaceChar(str.charAt(i))) {
				return false;
			}
		}
		return true;
	}

	public final void close() {
		System.out.println("SimplePostscript.close()");
		try {
			println("%%EOF\n");
			this.out.flush();
			this.out.close();
		} catch (Exception e) {
			System.out.println("SimplePostscript.close() failed - " + e);
		}
		this.isOpen = false;
	}
}
