// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * Class KColor holds the color constants and variables
 * used in hairSynth.
 * It also contains the method used to implement the
 * phong color calculation.
 */



package hairSynth;

import java.awt.Color;

public final class KColor {

	// ----- color choices
	private static final int red__COLOR = 0;
	// private static final int brownCOLOR = 1; // never read
	// private static final int blondCOLOR = 2; // never read
	// private static final int blackCOLOR = 3; // never read
	static final int cat5_COLOR = 4;
	static final int totalCOLORS = 5;
	static final String[] colorButtonStr = { "Red", "Brown", "Blonde", "Black", "Plastic" };

	// ----- background choices
	static final int blackBCKGRND = 0;
	// private static final int whiteBCKGRND = 1; // never read
	static final int totalBCKGRNDS = 2;
	static final String[] bckgrndButtonStr = { "Black background", "White background" };
	static final int[] colorBCKGRND = { 0, 255 };

	static int ixColrButOn = red__COLOR;
	static int ixBckgButOn = blackBCKGRND;

	// ----- cat5wire choices
	// private static final int red__CAT5 = 0; // never read
	// private static final int orangCAT5 = 1; // never read
	// private static final int greenCAT5 = 2; // never read
	// private static final int blue_CAT5 = 3; // never read
	private static final int whiteCAT5 = 4;
	static final int totalCAT5 = 5; // 5 colors

	static final float[] hueCAT5 = { 0.0f, 0.08333f, 0.41667f, 0.58333f, 0.0f };

	// ----- range
	private static final int cSHAD = 0; // in shade
	private static final int cMEDI = 1; // between shade and light
	private static final int cLITE = 2; // direct light; maximum luminosity

	// Usage example: KColor.hue[KColor.blackCOLOR][KColor.cmin]
	private static final float[][] cHue = { { 0.08f, 0.04f, 0.01f }, // red
			{ 0.05f, 0.06f, 0.08f }, // brown
			{ 0.05f, 0.08f, 0.08f }, // blonde
			{ 0.80f, 0.94f, 0.87f }, // black
			{ 0.0f, 0.0f, 0.0f } }; // cat5wire placeholder

	private static final float[][] cSat = { { 1.00f, 0.78f, 0.45f }, // red
			{ 0.61f, 0.48f, 0.45f }, // brown
			{ 0.56f, 0.28f, 0.10f }, // blonde
			{ 0.20f, 0.12f, 0.14f }, // black
			{ 0.0f, 0.0f, 0.0f } }; // cat5wire

	private static final float[][] cBri = { { 0.05f, 0.36f, 0.90f }, // red
			{ 0.10f, 0.37f, 0.77f }, // brown
			{ 0.38f, 0.79f, 0.92f }, // blonde
			{ 0.06f, 0.20f, 0.33f }, // black
			{ 0.0f, 0.0f, 0.0f } }; // cat5wire

	private KColor() {
		/* empty: DISABLE CONSTRUCTION */
	}

	static int getInverseOfCurrColorRGB() {

		float hue = cHue[KColor.ixColrButOn][cMEDI] + 0.5f; // 0.5f means 180 degree shift
		if (hue > 1.0f) {
			hue -= 1.0f;
		}
		float sat = 1.0f - cSat[KColor.ixColrButOn][cMEDI];

		// ------ returns rgb integer: (rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF
		return Color.HSBtoRGB(hue, sat, 1.0f);
	}

	static int getMedColor(final Strand strand) {

		float hue = 0.0f;
		float sat = 1.0f;
		float bri = 0.75f;

		if (KColor.ixColrButOn == cat5_COLOR) {

			int cat5color = strand.getCat5color();
			hue = hueCAT5[cat5color];
			if (cat5color == whiteCAT5) {
				sat = 0.0f;
			}

		} else {

			hue = cHue[KColor.ixColrButOn][cMEDI] + strand.getDeltaHue();
			// wrap hue around
			if (hue < 0.0f) {
				hue += 1.0f;
			} else if (hue > 1.0f) {
				hue -= 1.0f;
			}

			sat = cSat[KColor.ixColrButOn][cMEDI] + strand.getDeltaSat();
			// bounce sat
			if (sat < 0.0f) {
				sat = -sat;
			} else if (sat > 1.0f) {
				sat = 2.0f - sat;
			}

			bri = cBri[KColor.ixColrButOn][cMEDI] + strand.getDeltaBri();
			// bounce sat
			if (bri < 0.0f) {
				bri = -bri;
			} else if (bri > 1.0f) {
				bri = 2.0f - bri;
			}
		}

		// ------ returns rgb integer: (rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF
		return Color.HSBtoRGB(hue, sat, bri);
	}

	static int getPhongColor(final Strand parent_, final double strandDirAvg, final double elevation) {

		double facetZ = Math.sin(elevation);
		double facetYforZeroStrandNormal = Math.cos(elevation);
		double facetX = -facetYforZeroStrandNormal * Math.sin(strandDirAvg);
		double facetY = facetYforZeroStrandNormal * Math.cos(strandDirAvg);

		// the indented statements allow for arbitrary light angles
		// double lightX = 1.0;
		// double lightY = -1.0;
		// double lightZ = -1.0;
		// double cosAngle = K.cosAngle3D(lightX, lightY, lightZ, facetX, facetY, facetZ);
		// double zCross = lightX * facetY - lightY * facetX;

		double cosAngle = K.cosAngle3DtoRightLight(facetX, facetY, facetZ);
		double angle = Math.acos(cosAngle);

		// ---- change sign if negative CrossProduct for Z
		double zCross = facetY + facetX;
		if (zCross < 0.0) {
			angle = -angle;
		}

		boolean isInLight = true;
		if ((angle > (-K.PI05)) && (angle < K.PI05)) {
			isInLight = false; // meaning is in shade
		}

		// ---- start by assuming isShade = true
		float factorBri = 0.0f; // black

		if (isInLight) {
			factorBri = (float) Math.abs(cosAngle);
			// factorSat = 1.0f - factorBri; Keep as reference ... worked well
		}

		float hue = 0.0f;
		float sat = 0.75f;
		float bri = 0.75f;

		float shadSat = 1.0f;
		float liteSat = 0.25f;
		float shadBri = 0.25f;
		float liteBri = 1.0f;

		if (KColor.ixColrButOn == cat5_COLOR) {

			int cat5color = parent_.getCat5color();
			hue = hueCAT5[cat5color];
			if (cat5color == whiteCAT5) {
				sat = 0.0f;
				shadSat = 0.0f;
				liteSat = 0.0f;
				shadBri = 0.5f;
			}

		} else {

			hue = cHue[KColor.ixColrButOn][cMEDI] + parent_.getDeltaHue();
			sat = cSat[KColor.ixColrButOn][cMEDI] + parent_.getDeltaSat();
			bri = cBri[KColor.ixColrButOn][cMEDI] + parent_.getDeltaBri();

			if (hue < 0.0f) {
				hue += 1.0f;
			} else if (hue > 1.0f) {
				hue -= 1.0f;
			}
			sat = Math.max(0.0f, Math.min(1.0f, sat));
			bri = Math.max(0.0f, Math.min(1.0f, bri));

			// ----- calculate maximum shade and maximum light sat and bri, as needed
			if (factorBri < 0.5f) {
				shadSat = cSat[KColor.ixColrButOn][cSHAD]; // does not include Delta
				shadBri = cBri[KColor.ixColrButOn][cSHAD]; // therefore no need to notch
			} else if (factorBri > 0.5f) {
				liteSat = cSat[KColor.ixColrButOn][cLITE] + parent_.getDeltaSat();
				liteBri = cBri[KColor.ixColrButOn][cLITE] + parent_.getDeltaBri();
				liteSat = Math.max(0.0f, Math.min(1.0f, liteSat));
				liteBri = Math.max(0.0f, Math.min(1.0f, liteBri));
			}
		}

		// ----- modulate sat and bri with factorBri --------------------------
		if (factorBri < 0.5f) { // in shade
			sat = shadSat + (sat - shadSat) * 2.0f * factorBri;
			bri = shadBri + (bri - shadBri) * 2.0f * factorBri;
		} else if (factorBri > 0.5f) { // in light
			sat += (liteSat - sat) * 2.0f * (factorBri - 0.5f);
			bri += (liteBri - bri) * 2.0f * (factorBri - 0.5f);
		} // else if factorSat = 0.5 there is no change

		// ------ returns rgb integer: (rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF
		return Color.HSBtoRGB(hue, sat, bri);
	}

}
