// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * Class K holds most of the constants and shared (global)
 * variables used in hairSynth.
 * The sheer number of global variables is an indication
 * that more of them should have been packaged into classes
 * -- perhaps I will do that if time allows.
 * Class K also holds general utility methods usually
 * found in classes called "Util".
 */


package hairSynth;

import java.util.Random;

final class K {

	// ----- Actions
	static final int no_____ACTION = 0;
	static final int newStepACTION = 1;
	static final int presetsACTION = 2;
	static final int colors_ACTION = 3;
	static final int bckgrndACTION = 4;
	static final int reCentrACTION = 5;
	static final int zooSwtcACTION = 6;
	static final int dpi300_ACTION = 7;
	static final int savFileACTION = 8;

	static final String[] actionsSTR = { "no_____ACTION", "newStepACTION", "presetsACTION", "colors_ACTION",
			"bckgrndACTION", "reCentrACTION", "zooSwtcACTION", "dpi300_ACTION", "savFileACTION" };

	// ----- Button steps
	static final int shapeSTEP = 0;
	static final int colorSTEP = 1;
	static final int viewsSTEP = 2;
	static final int totalSTEPS = 3;

	static final String[] stepSTR = {"shapeSTEP", "colorSTEP", "viewsSTEP" };

	static final String[] stepButtonStr1 = { "               Step 1", "               Step 2", "               Step 3", };
	static final String[] stepButtonStr2 = { "           Set Shape", "    Select Color and", "   Render,  view and", };
	static final String[] stepButtonStr3 = { "", "    set zoom region", " write PostScript file", };

	// ----- presets
	static final int totalPRESETS = 6;
	static int ixPresetButOn = -1;
	private static final float[][] shapePRESETX = { //
	{ 240.0f, 240.0f, 240.0f, 790.0f }, // preset 0
			{ 790.0f, 10.0f, 10.0f, 790.0f }, // preset 1
			{ 240.0f, 240.0f, 240.0f, 790.0f }, // preset 2
			{ 790.0f, 10.0f, 790.0f, 790.0f }, // preset 3
			{ 10.0f, 790.0f, 10.0f, 790.0f }, // preset 4
			{ 150.0f, 150.0f, 790.0f, 790.0f } }; // preset 5

	private static final float[][] shapePRESETY = { //
	{ 570.0f, 400.0f, 200.0f, 400.0f }, // preset 0
			{ 400.0f, 790.0f, 790.0f, 400.0f }, // preset 1
			{ 790.0f, 130.0f, 10.0f, 400.0f }, // preset 2
			{ 215.0f, 790.0f, 215.0f, 400.0f }, // preset 3
			{ 400.0f, 590.0f, 400.0f, 400.0f }, // preset 4
			{ 240.0f, 650.0f, 240.0f, 400.0f } }; // preset 5

	// ----- shared state variables
	static int ixStepButOn = shapeSTEP;

	static boolean isReCenterOn = false;

	// ----- index for View Buffers ... also used for Buttons
	static final int ixViewFull1x = 0;
	static final int ixViewZoom3x = 1;
	static final int ixViewZoo10x = 2;
	static final int ixViewZoo30x = 3; // not read
	static final int ixViewZo100x = 4;
	static final int ixViewSwat3x = 5;
	static final int ixViewSwa10x = 6;
	static final int ixViewSwa30x = 7;
	static final int ixViewSw100x = 8;
	static final int ixViewTOTAL = 9;

	static final float[] zoomFACTOR = { 1.0f, 3.0f, 10.0f, 30.0f, 100.0f, 3.0f, 10.0f, 30.0f, 100.0f };
	// clipFACTOR is used to clip segments ...
	static final float[] clipFACTOR = { 1.0f, 3.0f, 10.0f, 30.0f, 100.0f, 4.0f, 15.0f, 50.0f, 200.0f };
	static final double minSegmentLENGTH = 0.0005; // 1.0 / (300.0 * 8.0);// about 1.0 DPI at 300 DPI
	// maxSegmentLENGTH afffects precision and look of swatches. It was 75.0 / (300.0 * 8.0); // about 1/4 of an inch
	// segment lengths for swatch and zoom need to be the same so that drawings overlap
	static final double[] maxSegmentLENGTH = { 0.05, 0.03, 0.02, 0.005, 0.002, 0.03, 0.02, 0.005, 0.002 };

	static final String[] zoomButStr = { "No zoom", "Zoom 3x", "Zoom 10x", "Zoom 30x", "Zoom 100x" };
	static final String[] zoomFileStr = { "Full1x", "Zoom3x", "Zoo10x", "Zoo30x", "Zo100x", "Swat3x", "Swa10x",
			"Swa30x", "Sw100x" };

	static int ixZoomSwatchButOn = ixViewFull1x; // index combines two buttons

	static boolean isDPI300on = false;

	static final int inactiveSAVEFILE = 0;
	static final int readyToSAVEFILE = 1;
	static final int performingSAVEFILE = 2;
	static final int doneSAVEFILE = 3;
	static int saveFileState = inactiveSAVEFILE;

	static int currentChildBeingDrawn = 0;

	// ------------------------------------------------------------------------
	// ----- dash and canvas dimensions
	static final float[] DASH_WIDTH = {200.0f, 200.0f, 240,0f}; // leave the same for applet and applic
	static final float[] STEP_BUT_AREA_HEIGHT = {270.0f, 340.0f, 444.0f};
	static final float[] CANVAS_SCALE = {640.0f, 800.0f, 1024.f}; // size of canvas
	static float canvasScale;
	private static float canvasCenter;
	// my convention is Bottom Left corner is [0,0]

	static final float diskRADIUS = 10;
	static final float outOfCANVAS = 10000.0f; // arbitrary large distance


	// ----- shared ZoomCenter
	private static final float canvasBORDER = 0.10f;
	static final float borderReductionFACTOR = 1.0f - 2.0f * canvasBORDER;
	static final float easing = 0.333f; // easing constant for boxCenter
	static float scaleToPixEased = CANVAS_SCALE[1] * borderReductionFACTOR;
	static float scaleToPixBunch = scaleToPixEased;

	static float boxSinglHlfS = 0.5f; // set in easeCanvas.....
	static float boxSinglCtrX = 0.5f; // shared among all Buffers
	static float boxSinglCtrY = 0.5f;

	static float boxBunchHlfS = boxSinglHlfS;
	static float boxBunchCtrX = boxSinglCtrX;
	static float boxBunchCtrY = boxSinglCtrY;

	static float zoomCtrX;
	static float zoomCtrY;

	// Human hair thickness is 60 to 90 microns in European populations and up to 120 in Asian
	// I have used 100 microns = 0.004 inch
	// Canvas width in screen = 800 pix / 90 dpi = approx 8 inches (same as A4 width used in PS file)
	// Hair thickness expressed in Canvas units (0..1) is therefore 0.004 / 8 = 0.0005
	static final float strandHalfWIDTH = 0.00025f; // expressed in Canvas units with no zoom


	// ----- postScript document scale and dimensions
	static final float DOC_SCALE = 604.0f; // scales 1.0 to doc_width
	static final float DOC_CENTER = DOC_SCALE / 2.0f;
	static float docScaleToCanvasScale;

	static float scaleToPs = scaleToPixBunch * docScaleToCanvasScale;

	static final int ixOscBLU0 = 0; // beginning
	static final int ixOscMAG1 = 1; // both
	static final int ixOscRED2 = 2; // end
	static final int ixOscTOT3 = 3;
	static final int ixDirGRN3 = 3;
	static final int totDISKS4 = 4;

	static int[] diskParamCodeInt = new int[totDISKS4]; // retains current params for disk
	static long oscParamCodeLong = 0; // use zero as arbitrary initial code

	// ----- dash and canvas position
	static boolean isDashOnRight = false;
	static float dash_x0 = 0.0f;
	static float canvas_x0;

	static final double totalStrandLEN = 2.0;
	static double distZOOM = 0.67 * totalStrandLEN;

	static final double face1xINIT = 0.0;
	static final double face1yINIT = 1.0;
	// Normal hair density is 100 folicles per sq cm or 1000 for radius of 0.7 inch
	// ... expressed in canvas units and ellyptical tuft ... used higher density
	static final double radiusXtuftROOT = 0.03;
	static final double radiusYtuftROOT = 0.03;

	// ----- math constants
	static final double PI1 = Math.PI;
	static final double PI2 = 2.0 * Math.PI;
	static final double PI05 = Math.PI / 2.0;
	static final double PI025 = Math.PI / 4.0;
	static final double toRADIANS = PI2 / 360.0;
	static final double BIGNUM = 1000000000.0;

	static final float PI1fl = (float) PI1;
	static final float PI2fl = (float) PI2;

	static final float LN4 = (float) Math.log(4.0);
	static final float LN04 = (float) Math.log(0.4);

	static Random r = new Random();

	private K() {
		/* empty: DISABLE CONSTRUCTION */
	}

	static void init() {
		canvasScale = CANVAS_SCALE[Main.ixSIZE];
		canvasCenter = CANVAS_SCALE[Main.ixSIZE] / 2.0f;
		docScaleToCanvasScale = DOC_SCALE / canvasScale;

		canvas_x0 = DASH_WIDTH[Main.ixSIZE];
	}

	///////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////

	static float shapePresetX(final int ixPreset, final int ixDisk) {

		return scalePreset( shapePRESETX[ixPreset][ixDisk] );
	}

	static float shapePresetY(final int ixPreset, final int ixDisk) {

		return scalePreset( shapePRESETY[ixPreset][ixDisk] );
	}

	private static float scalePreset(final float preset) {

		if (Main.ixSIZE == 1) {
			return preset;
		}
		float canvasCtrThis = CANVAS_SCALE[Main.ixSIZE] / 2.0f ;
		float canvasCtrLarge = CANVAS_SCALE[1] / 2.0f ;
		float scale = (canvasCtrThis - diskRADIUS) / (canvasCtrLarge - diskRADIUS);
		return (canvasCtrThis + (preset - canvasCtrLarge) * scale);
	}

	///////////////////////////////////////////////////////////////////////////
	///////////////////////////////////////////////////////////////////////////

	static float randomInRangeFl(final float range) {
		return range * (r.nextFloat() - 0.5f);
	}

	static void setCtrZoomXandY(final float zoomCtrXpix, final float zoomCtrYpix) {

		zoomCtrX = boxBunchCtrX + (zoomCtrXpix - canvasCenter) / scaleToPixBunch;
		zoomCtrY = boxBunchCtrY - (zoomCtrYpix - canvasCenter) / scaleToPixBunch;
	}

	static boolean setBoxSingleXYandScaleAndReturnIsEasing(final float tmpBoxLft, final float tmpBoxRgt,
			final float tmpBoxBot, final float tmpBoxTop) {

		float boxHalfSide = tmpBoxRgt - tmpBoxLft;
		if (boxHalfSide < (tmpBoxTop - tmpBoxBot)) {
			boxHalfSide = tmpBoxTop - tmpBoxBot;
		}
		boxHalfSide = Math.max(0.000001f, boxHalfSide / 2.0f); // avoid divisions by zero in scaling

		float newX = (tmpBoxRgt + tmpBoxLft) / 2.0f;
		float newY = (tmpBoxBot + tmpBoxTop) / 2.0f;

		boxSinglHlfS = boxHalfSide; // update boxHalfSide
		float scaleToPixSingleExact = canvasScale * borderReductionFACTOR * 0.5f / boxHalfSide; // only place where this is set

		boolean isEasing = Math.abs(scaleToPixSingleExact - scaleToPixEased) > 0.1f;

		if (isEasing) { // tolerance
			boxSinglCtrX = (1.0f - easing) * boxSinglCtrX + easing * newX;
			boxSinglCtrY = (1.0f - easing) * boxSinglCtrY + easing * newY;
			scaleToPixEased = (1.0f - easing) * scaleToPixEased + easing
					* scaleToPixSingleExact;
		} else {
			boxSinglCtrX = newX;
			boxSinglCtrY = newY;
			scaleToPixEased = scaleToPixSingleExact;
		}

		boxBunchHlfS = boxSinglHlfS;
		boxBunchCtrX = boxSinglCtrX;
		boxBunchCtrY = boxSinglCtrY;

		scaleToPixBunch = scaleToPixSingleExact;
		scaleToPs = scaleToPixBunch * docScaleToCanvasScale;

		return isEasing;
	}

	static void setBoxBunchTmpXYandScaleIfLargerBoudariesThanSingle(final float tmpBoxLft, final float tmpBoxRgt,
			final float tmpBoxBot, final float tmpBoxTop) {

		float currBoxLft = boxBunchCtrX - boxBunchHlfS;
		float currBoxRgt = boxBunchCtrX + boxBunchHlfS;
		float currBoxBot = boxBunchCtrY - boxBunchHlfS;
		float currBoxTop = boxBunchCtrY + boxBunchHlfS;

		if ((tmpBoxLft < currBoxLft) || (tmpBoxRgt > currBoxRgt) || (tmpBoxBot < currBoxBot)
				|| (tmpBoxTop > currBoxTop)) {

			float boxHalfSide = tmpBoxRgt - tmpBoxLft;
			if (boxHalfSide < (tmpBoxTop - tmpBoxBot)) {
				boxHalfSide = tmpBoxTop - tmpBoxBot;
			}

			boxBunchHlfS = Math.max(0.000001f, boxHalfSide / 2.0f); // avoid divisions by zero in scaling
			boxBunchCtrX = (tmpBoxRgt + tmpBoxLft) / 2.0f;
			boxBunchCtrY = (tmpBoxBot + tmpBoxTop) / 2.0f;

			scaleToPixBunch = canvasScale * borderReductionFACTOR * 0.5f / boxBunchHlfS;
			scaleToPs = scaleToPixBunch * docScaleToCanvasScale;
		}
	}

	static float canvasSinglXpixEased(final float x) {
		return (canvasCenter + scaleToPixEased * (x - boxSinglCtrX));
	}

	static float canvasSinglYpixEased(final float y) {
		return (canvasCenter - scaleToPixEased * (y - boxSinglCtrY));
	}

	static float canvasCenter(final float x) {
		return (canvasCenter + scaleToPixBunch * (x - boxBunchCtrX));
	}

	static float canvasBunchYpix(final float y) {
		return (canvasCenter - scaleToPixBunch * (y - boxBunchCtrY));
	}

	static float canvasZoomXpix(final float x, final int ixView) {
		return (canvasCenter + scaleToPixBunch * (x - zoomCtrX) * zoomFACTOR[ixView]);
	}

	static float canvasZoomYpix(final float y, final int ixView) {
		return (canvasCenter - scaleToPixBunch * (y - zoomCtrY) * zoomFACTOR[ixView]);
	}

	static float psZoomXpix(final float x, final int ixView) {
		return (DOC_CENTER + scaleToPs * (x - zoomCtrX) * zoomFACTOR[ixView]);
	}

	static float psZoomYpix(final float y, final int ixView) {
		return (DOC_CENTER - scaleToPs * (y - zoomCtrY) * zoomFACTOR[ixView]);
	}

	static float getZoomBoxHalfSide(final int ixView) {
		return 0.5f / zoomFACTOR[ixView];
	}

	static boolean isInZoomBox(final float x_, final float y_, final int ixView) {
		// note widening of box by strandHalfWIDTH ... also note adjusting for borderReductionFACTOR
		float zoomHalfSide = strandHalfWIDTH + boxBunchHlfS / (clipFACTOR[ixView] * borderReductionFACTOR);
		return (((x_ > (zoomCtrX - zoomHalfSide)) && (x_ < (zoomCtrX + zoomHalfSide))) //
				&& ((y_ > (zoomCtrY - zoomHalfSide)) && (y_ < (zoomCtrY + zoomHalfSide))));
	}

	static int numberOfFacetsResultingInFacetWidthOfOnePixel() {

		// hair thickness * scale * zoomFactor / facets = 1 pix ... therefore facets = ... using 100 DPI
		int facets = (int) (0.5f + 2.0f * strandHalfWIDTH * scaleToPixBunch * zoomFACTOR[K.ixZoomSwatchButOn]);
		if (isDPI300on) {
			facets *= 3; // calculates facets based on 300 dpi instead of 100 dpi (800 dots = 8 inches)
		}

		return Math.max(1, Math.min(BufferView.maxFACETS, facets));
	}

	// /////////////////////////////////////////////////////////////////////////
	// \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
	// /////////////////////////////////////////////////////////////////////////

	static double forceFromMinusPiToPlusPi(final double face1N) {

		double normalized_ = face1N;
		// -----Normalizing face1Normal to -Pi..+Pi
		while (normalized_ > PI1) { // max is 180 degrees
			normalized_ -= PI2;
		}
		while (normalized_ < (-PI1)) { // min is -180 degrees
			normalized_ += PI2;
		}
		return normalized_;
	}

	static double bisect(final float a0, final float a1) {
		// this only works with normalized angles ... see forceFromMinusPiToPlusPi

		double bisect_ = (a0 + a1) / 2.0;
		// ------ make sure use bisection of smaller angle when diff signs
		if ((a0 > 0) && (a1 < 0) && ((a0 - a1) > K.PI1fl)) {

			if (bisect_ < 0.0) {
				bisect_ += Math.PI;
			} else {
				bisect_ -= Math.PI;
			}

		} else if ((a1 > 0) && (a0 < 0) && ((a1 - a0) > K.PI1fl)) {

			if (bisect_ < 0.0) {
				bisect_ += Math.PI;
			} else {
				bisect_ -= Math.PI;
			}
		}
		return bisect_;
	}

	// /////////////////////////////////////////////////////////////////////////
	// \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\
	// /////////////////////////////////////////////////////////////////////////

	static double angle3D(final double x0, final double y0, final double z0, final double x1, final double y1,
			final double z1) {

		double len0 = Math.sqrt(x0 * x0 + y0 * y0 + z0 * z0);
		double len1 = Math.sqrt(x1 * x1 + y1 * y1 + z1 * z1);
		double len0xlen1 = len0 * len1;
		if (len0xlen1 > 0.0) {
			double dot = x0 * x1 + y0 * y1 + z0 * z1;
			double cosA = dot / len0xlen1;
			return Math.acos(cosA);
		}
		return 0.0; // indeterminate
	}

	static double cosAngle3D(final double x0, final double y0, final double z0, final double x1, final double y1,
			final double z1) {

		double len0 = Math.sqrt(x0 * x0 + y0 * y0 + z0 * z0);
		double len1 = Math.sqrt(x1 * x1 + y1 * y1 + z1 * z1);
		double len0xlen1 = len0 * len1; // always positive
		if (len0xlen1 > 0.0) {
			double dot = x0 * x1 + y0 * y1 + z0 * z1;
			return dot / len0xlen1;
		}
		return 0.0; // indeterminate
	}

	static double angle3DtoLeftLight(final double x0, final double y0, final double z0) {

		// light x1=1..from left y1=-1..from top, z1=-1..from back

		double len0 = Math.sqrt(x0 * x0 + y0 * y0 + z0 * z0);
		double len0xlen1 = len0 * 1.732051; // sqrt(3.0)
		double dot = x0 - y0 - z0;
		double cosA = dot / len0xlen1;
		return Math.acos(cosA);
	}

	static double cosAngle3DtoLeftLight(final double x0, final double y0, final double z0) {

		// light x1=1..from left y1=-1..from top, z1=-1..from back

		double len0 = Math.sqrt(x0 * x0 + y0 * y0 + z0 * z0);
		double len0xlen1 = len0 * 1.732051; // sqrt(3.0)
		double dot = x0 - y0 - z0;
		return dot / len0xlen1;
	}

	static double cosAngle3DtoRightLight(final double x0, final double y0, final double z0) {

		// light x1=-1..from left y1=-1..from top, z1=-1..from back

		double len0 = Math.sqrt(x0 * x0 + y0 * y0 + z0 * z0);
		double len0xlen1 = len0 * 1.732051; // sqrt(3.0)
		double dot = -x0 - y0 - z0;
		return dot / len0xlen1;
	}
}
