// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * Class DiskCanvas holds the methods necessary to
 * translate mouse gestures into disk positions.
 * Disk positions are used to record the parameters
 * used to set the shape the master hair strand.
 */

package hairSynth;

import processing.core.PApplet;

final class DiskCanvas {

	// All vars private
	private float pressCanvasX;
	private float pressCanvasY;

	private Disk[] disk = new Disk[K.totDISKS4];
	private int activeDisk = -1; // if negative, then no disk is active
	private float activeDiskX0;
	private float activeDiskY0;
	private float lastDragDX;
	private float lastDragDY;

	private PApplet papplet;

	DiskCanvas(final PApplet papplet_) {
		this.papplet = papplet_;

		for (int ixDisk = 0; ixDisk < K.totDISKS4; ixDisk++) {
			float paramX = K.shapePresetX(K.totalPRESETS - 1,ixDisk);
			float paramY = K.shapePresetY(K.totalPRESETS - 1,ixDisk);
			this.disk[ixDisk] = new Disk(papplet_, ixDisk, paramX, paramY);
		}
	}

	float getActiveX() {
		return this.disk[this.activeDisk].getX();
	}

	float getActiveY() {
		return this.disk[this.activeDisk].getY();
	}

	void setDiskXY(final int ixDisk, final float x_, final float y_) {
		this.disk[ixDisk].setX(x_);
		this.disk[ixDisk].setY(y_);
	}

	void printParams() {
		System.out.println("X 0123: " + (int) (0.5 + this.disk[0].getX()) + " " + (int) (0.5 + this.disk[1].getX())
				+ " " + (int) (0.5 + this.disk[2].getX()) + (int) (0.5 + this.disk[3].getX()) + "  Y 0123: "
				+ (int) (0.5 + this.disk[0].getY()) + " " + (int) (0.5 + this.disk[1].getY()) + " "
				+ (int) (0.5 + this.disk[2].getY()) + " " + (int) (0.5 + this.disk[3].getY()));
	}

	boolean testHoverOverDisks(final float mouseX, final float mouseY) {

		boolean isChangedHover = false;
		float smallestDist = K.outOfCANVAS;
		int closestDisk = 0;

		for (int ixDisk = 0; ixDisk < K.totDISKS4; ixDisk++) {

			this.disk[ixDisk].setIsActive(false);
			float dist = this.disk[ixDisk].distToCenter(mouseX - K.canvas_x0, mouseY);
			if (smallestDist > dist) {
				smallestDist = dist;
				closestDisk = ixDisk;
			}
		}

		if (smallestDist <= K.diskRADIUS) {
			this.disk[closestDisk].setIsActive(true);
			if (this.activeDisk != closestDisk) {
				this.activeDisk = closestDisk;
				isChangedHover = true; // change in color
			}
		} else { // not hovering over any of the disks
			if (this.activeDisk > -1) {
				this.activeDisk = -1;
				isChangedHover = true; // change in color
			}
		}
		return isChangedHover;
	}

	boolean recordActiveDiskPositionOnPressIfActive(final float mouseX, final float mouseY) {

		this.lastDragDX = 0.0f;
		this.lastDragDY = 0.0f;

		if (this.activeDisk == -1) {
			this.testHoverOverDisks(mouseX, mouseY);
		}

		if (this.activeDisk > -1) {

			this.pressCanvasX = mouseX - K.canvas_x0;
			this.pressCanvasY = mouseY;

			this.activeDiskX0 = this.disk[this.activeDisk].getX();
			this.activeDiskY0 = this.disk[this.activeDisk].getY();
			return true;
		}
		return false;
	}

	int moveActiveDiskPositionOnDrag(final float mouseX, final float mouseY) {

		if (this.activeDisk > -1) { // only record drag only if there is an active disk

			float dragDX = mouseX - K.canvas_x0 - this.pressCanvasX;
			float dragDY = mouseY - this.pressCanvasY;

			if ((this.lastDragDX != dragDX) || (this.lastDragDY != dragDY)) { // dragged

				this.lastDragDX = dragDX;
				this.lastDragDY = dragDY;

				if (this.activeDisk < K.ixDirGRN3) { // ignore DX for green disk
				this.disk[this.activeDisk].setX(this.activeDiskX0 + this.lastDragDX);
				}
				this.disk[this.activeDisk].setY(this.activeDiskY0 + this.lastDragDY);

				return this.activeDisk;
			}
		}
		return -1;
	}

	void draw() {

		this.papplet.noStroke();

		for (int ixDisk = 0; ixDisk < K.totDISKS4; ixDisk++) {
			this.disk[ixDisk].draw();
		}
	}
}
