// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * class BufferView includes the methods that draw
 * the hair strands on Step 3.
 * It also includes the method that writes polygons
 * to the vector file.
 */

package hairSynth;

import processing.core.PApplet;
import processing.core.PConstants;

final class BufferView extends Buffer {

	// NOTE: all references to pg (instance of pGraphics) are conditional to isAPPLIC = true
	// This is a work-around as I could not get pg.image to work for an element of an array
	// No public vars here
	private int myID; // index of bufferView array, set at creation. From 0..3 now
	private FilePS ps;

	private long lastParamCodeDrawn = K.oscParamCodeLong;
	private int ixLastColorDrawn = KColor.ixColrButOn;
	private int ixLastBckgrDrawn = KColor.ixBckgButOn;
	private float lastZoomCtrXdrawn = K.zoomCtrX;
	private float lastZoomCtrYdrawn = K.zoomCtrY;
	private int lastFacetsOnDrawn = 0;

	private int ixCurrChild;

	private int firstThisBunch;
	private int firstNextBunch;
	private static int strandsBetweenRedraws = 16; // 32 results in about 10 FPS in my computer

	private float facetElevation = (float) (K.PI1 / 3.0);

	static final int maxFACETS = 40;
	private float[] ptFace0x = new float[maxFACETS + 1];
	private float[] ptFace0y = new float[maxFACETS + 1];
	private float[] ptFace1x = new float[maxFACETS + 1];
	private float[] ptFace1y = new float[maxFACETS + 1];

	private float[] psFace0x = new float[maxFACETS + 1];
	private float[] psFace0y = new float[maxFACETS + 1];
	private float[] psFace1x = new float[maxFACETS + 1];
	private float[] psFace1y = new float[maxFACETS + 1];

	private int actualFacets = maxFACETS;

	private BufferBunch bufferBunch; // never created, just holds pointer

	BufferView(final PApplet papplet_, final FilePS ps_, final BufferBunch bufferBunch_, final int myID_) {

		if (Main.isAPPLIC) {
			createPG(papplet_);
			this.ps = ps_; // just pointer
		} else {
			this.papplet = papplet_; // needs to set if not set in Buffer
		}
		this.bufferBunch = bufferBunch_; // just pointer
		this.myID = myID_;
	}

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	boolean populateAndRender(final boolean isWorkedThisCycle) {

		boolean isWorkedThisMethod = true;

		if (this.bufferBunch.getIsParamsOrColorsNeedPopulating()) {
			this.stage = Buffer.stagePOPULATE0;
		} else if ((this.lastParamCodeDrawn != K.oscParamCodeLong) || (this.ixLastColorDrawn != KColor.ixColrButOn)
				|| (this.ixLastBckgrDrawn != KColor.ixBckgButOn) || (this.lastZoomCtrXdrawn != K.zoomCtrX)
				|| (this.lastZoomCtrYdrawn != K.zoomCtrY)
				|| (this.lastFacetsOnDrawn != K.numberOfFacetsResultingInFacetWidthOfOnePixel())) {
			// ----- changed params, color scheme or background
			if (this.stage > Buffer.stageINITDRAW1) {
				this.stage = Buffer.stageINITDRAW1; // redraw
			}
		}

		switch (this.stage) {

		case Buffer.stagePOPULATE0:

			this.bufferBunch.populateParamsOrColors();

			if (this.bufferBunch.getIsParamsAndColorsArePopulated()) {
				this.stage = Buffer.stageINITDRAW1;
			}

			break;

		case Buffer.stageINITDRAW1:

			if (Main.isAPPLIC) {
				this.clearPgScreen();
				this.copyToScreen();
			} else {
				this.clearPAppletScreen();
			}

			this.actualFacets = K.numberOfFacetsResultingInFacetWidthOfOnePixel();

			this.lastParamCodeDrawn = K.oscParamCodeLong;
			this.ixLastColorDrawn = KColor.ixColrButOn;
			this.ixLastBckgrDrawn = KColor.ixBckgButOn;
			this.lastZoomCtrXdrawn = K.zoomCtrX;
			this.lastZoomCtrYdrawn = K.zoomCtrY;
			this.lastFacetsOnDrawn = this.actualFacets;

			if (Main.isAPPLIC) {
				this.ps.OpenIfNecessary();
			}
			this.firstNextBunch = 0;
			this.stage = Buffer.stageDRAWABUNCH2;

			break;

		case Buffer.stageDRAWABUNCH2: // Several Passes

			this.firstThisBunch = this.firstNextBunch;
			this.firstNextBunch += strandsBetweenRedraws;

			if ((!Main.isAPPLIC) & (this.actualFacets > 3)) {
				this.papplet.noSmooth();
			}

			this.ixCurrChild = 0; // incremented before using
			drawSectionOfTreeAndReturnIsDone(maxLEVEL, 0);

			if (Main.isAPPLIC) {
				this.copyToScreen();
			} else {
				this.papplet.smooth();
			}

			if (this.firstNextBunch > numberOfStrandsTRAVERSED) {
				this.stage = Buffer.stageDONE4;
			}
			break;

		case Buffer.stageREFRESHSCREEN3: // used when done but changed button choice

			if (Main.isAPPLIC) {
				this.copyToScreen();
			}

			this.stage = Buffer.stageDONE4;
			break;

		case Buffer.stageDONE4:

			K.currentChildBeingDrawn = 0;
			if (Main.isAPPLIC) {
				this.ps.closeIfNecessary();
			}
			isWorkedThisMethod = false;
			break;

		default:
			isWorkedThisMethod = false;
			break;
		}

		return (isWorkedThisCycle || isWorkedThisMethod);
	}

	// =========================================================================
	// ALL PRIVATE BELOW
	// =========================================================================

	private void drawSectionOfTreeAndReturnIsDone(final int level_, final int idParent_) {

		if (level_ > 0) {

			this.ixCurrChild++;
			drawSectionOfTreeAndReturnIsDone(level_ - 1, this.ixCurrChild);

			this.ixCurrChild++;
			drawSectionOfTreeAndReturnIsDone(level_ - 1, this.ixCurrChild);

		} else if ((idParent_ >= this.firstThisBunch) && (idParent_ < this.firstNextBunch)) { // Level = 0
			drawStrand(idParent_);
		}
	}

	private void drawStrand(final int idParent_) {

		K.currentChildBeingDrawn = idParent_;
		Strand parent_ = this.bufferBunch.getStrand(idParent_);

		if (Main.isAPPLIC) {
			this.pg.beginDraw();
		}

		double dist = 0.0;

		while (dist < K.totalStrandLEN) {

			dist = parent_.nextSegment(dist);

			float face0X = parent_.getFace0Xfloat();
			float face0Y = parent_.getFace0Yfloat();
			float face1X = parent_.getFace1Xfloat();
			float face1Y = parent_.getFace1Yfloat();

			if ((this.myID == 0) || K.isInZoomBox(face0X, face0Y, this.myID)
					|| K.isInZoomBox(face1X, face1Y, this.myID)) {

				float face0Normal = parent_.getFace0NormalFloat();
				float face1Normal = parent_.getFace1NormalFloat();

				double strandDirAvg = K.bisect(face0Normal, face1Normal);

				if (this.actualFacets == 1) {

					int rgb = KColor.getPhongColor(parent_, strandDirAvg, this.facetElevation);

					if (Main.isAPPLIC) {
						this.pg.stroke((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

						if (K.ixZoomSwatchButOn == K.ixViewFull1x) { // distinction just for speed
							this.pg.line(K.canvasCenter(face0X), K.canvasBunchYpix(face0Y), K.canvasCenter(face1X), K
									.canvasBunchYpix(face1Y));
						} else {
							this.pg.line(K.canvasZoomXpix(face0X, K.ixZoomSwatchButOn), K.canvasZoomYpix(face0Y,
									K.ixZoomSwatchButOn), K.canvasZoomXpix(face1X, K.ixZoomSwatchButOn), K
									.canvasZoomYpix(face1Y, K.ixZoomSwatchButOn));
						}
					} else {
						this.papplet.stroke((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

						if (K.ixZoomSwatchButOn == K.ixViewFull1x) { // distinction just for speed
							this.papplet.line(K.canvas_x0 + K.canvasCenter(face0X), K.canvasBunchYpix(face0Y),
									K.canvas_x0 + K.canvasCenter(face1X), K.canvasBunchYpix(face1Y));
						} else {
							this.papplet.line(K.canvas_x0 + K.canvasZoomXpix(face0X, K.ixZoomSwatchButOn), K
									.canvasZoomYpix(face0Y, K.ixZoomSwatchButOn), K.canvas_x0
									+ K.canvasZoomXpix(face1X, K.ixZoomSwatchButOn), K.canvasZoomYpix(face1Y,
									K.ixZoomSwatchButOn));
						}
					}
				}
				if ((this.actualFacets > 1) || (K.saveFileState == K.performingSAVEFILE)) {

					this.drawSegmentNfacets(parent_, face0X, face0Y, face0Normal, face1X, face1Y, face1Normal,
							strandDirAvg);
				}
			}
		}
		if (Main.isAPPLIC) {
			this.pg.endDraw();
		}
	}

	private void drawSegmentNfacets(final Strand parent_, final float face0X, final float face0Y,
			final float face0Normal, final float face1X, final float face1Y, final float face1Normal,
			final double strandDirAvg) {

		if (Main.isAPPLIC) {
			this.pg.noStroke();
		} else {
			this.papplet.noStroke();
		}

		float sinNormal0xHWidth = K.strandHalfWIDTH * (float) Math.sin(face0Normal);
		float cosNormal0xHWidth = K.strandHalfWIDTH * (float) Math.cos(face0Normal);

		float sinNormal1xHWidth = K.strandHalfWIDTH * (float) Math.sin(face1Normal);
		float cosNormal1xHWidth = K.strandHalfWIDTH * (float) Math.cos(face1Normal);

		// Segment; to the left is face0; to the right is face1
		// x0E --- x1E
		// |........|
		// x0A --- x1A

		// ----- populate array used to draw on canvas
		// ----- set extreme vertexes
		int ixLast = this.actualFacets; // number of facets
		int ixFrst = 0; // interchangeable with zero
		this.ptFace0x[ixLast] = K.canvasZoomXpix(face0X - sinNormal0xHWidth, this.myID);
		this.ptFace0y[ixLast] = K.canvasZoomYpix(face0Y + cosNormal0xHWidth, this.myID);
		this.ptFace0x[ixFrst] = K.canvasZoomXpix(face0X + sinNormal0xHWidth, this.myID);
		this.ptFace0y[ixFrst] = K.canvasZoomYpix(face0Y - cosNormal0xHWidth, this.myID);

		this.ptFace1x[ixLast] = K.canvasZoomXpix(face1X - sinNormal1xHWidth, this.myID);
		this.ptFace1y[ixLast] = K.canvasZoomYpix(face1Y + cosNormal1xHWidth, this.myID);
		this.ptFace1x[ixFrst] = K.canvasZoomXpix(face1X + sinNormal1xHWidth, this.myID);
		this.ptFace1y[ixFrst] = K.canvasZoomYpix(face1Y - cosNormal1xHWidth, this.myID);

		// ----- set intermediary vertexes
		for (int ixFacet = ixFrst + 1; ixFacet < ixLast; ixFacet++) { // no need to redo extremes
			float weightLast = (float) (ixFacet - ixFrst) / ixLast;
			float weigthFrst = (float) (ixLast - ixFacet) / ixLast;
			this.ptFace0x[ixFacet] = weightLast * this.ptFace0x[ixLast] + weigthFrst * this.ptFace0x[ixFrst];
			this.ptFace0y[ixFacet] = weightLast * this.ptFace0y[ixLast] + weigthFrst * this.ptFace0y[ixFrst];
			this.ptFace1x[ixFacet] = weightLast * this.ptFace1x[ixLast] + weigthFrst * this.ptFace1x[ixFrst];
			this.ptFace1y[ixFacet] = weightLast * this.ptFace1y[ixLast] + weigthFrst * this.ptFace1y[ixFrst];
		}

		// --------------------------------------------------------------------
		// ----- populate array used to save file, if necessary
		if (K.saveFileState == K.performingSAVEFILE) {
			this.psFace0x[ixLast] = K.psZoomXpix(face0X - sinNormal0xHWidth, this.myID);
			this.psFace0y[ixLast] = K.psZoomYpix(face0Y + cosNormal0xHWidth, this.myID);
			this.psFace0x[ixFrst] = K.psZoomXpix(face0X + sinNormal0xHWidth, this.myID);
			this.psFace0y[ixFrst] = K.psZoomYpix(face0Y - cosNormal0xHWidth, this.myID);

			this.psFace1x[ixLast] = K.psZoomXpix(face1X - sinNormal1xHWidth, this.myID);
			this.psFace1y[ixLast] = K.psZoomYpix(face1Y + cosNormal1xHWidth, this.myID);
			this.psFace1x[ixFrst] = K.psZoomXpix(face1X + sinNormal1xHWidth, this.myID);
			this.psFace1y[ixFrst] = K.psZoomYpix(face1Y - cosNormal1xHWidth, this.myID);

			// ----- set intermediary vertexes
			for (int ixFacet = ixFrst + 1; ixFacet < ixLast; ixFacet++) { // no need to redo extremes
				float weightLast = (float) (ixFacet - ixFrst) / ixLast;
				float weigthFrst = (float) (ixLast - ixFacet) / ixLast;
				this.psFace0x[ixFacet] = weightLast * this.psFace0x[ixLast] + weigthFrst * this.psFace0x[ixFrst];
				this.psFace0y[ixFacet] = weightLast * this.psFace0y[ixLast] + weigthFrst * this.psFace0y[ixFrst];
				this.psFace1x[ixFacet] = weightLast * this.psFace1x[ixLast] + weigthFrst * this.psFace1x[ixFrst];
				this.psFace1y[ixFacet] = weightLast * this.psFace1y[ixLast] + weigthFrst * this.psFace1y[ixFrst];
			}
		}

		// Facet vertices
		// x0E --- x1E ... already calculated
		// x0D --- x1D
		// x0C --- x1C
		// x0B --- x1B
		// x0A --- x1A ... already calculated

		// --------------------------------------------------------------------
		// ----- paint facet on buffer
		for (int ixFacet = ixFrst; ixFacet < ixLast; ixFacet++) {

			double elevation = K.PI025; // if only one facet, make it 45%
			if (this.actualFacets > 1) {
				elevation = (2.0 * (ixLast - ixFacet) - 1.0) * K.PI05 / ixLast;
			}
			int rgb = KColor.getPhongColor(parent_, strandDirAvg, elevation);

			if (this.actualFacets > 1) { // if equal to 1 then already drawn line
				if (Main.isAPPLIC) {
					this.pg.fill((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

					this.pg.beginShape();
					this.pg.vertex(this.ptFace0x[ixFacet + 1], this.ptFace0y[ixFacet + 1]);
					this.pg.vertex(this.ptFace0x[ixFacet], this.ptFace0y[ixFacet]);
					this.pg.vertex(this.ptFace1x[ixFacet], this.ptFace1y[ixFacet]);
					this.pg.vertex(this.ptFace1x[ixFacet + 1], this.ptFace1y[ixFacet + 1]);
					this.pg.endShape(PConstants.CLOSE);
				} else {
					this.papplet.fill((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

					this.papplet.beginShape();
					this.papplet.vertex(K.canvas_x0 + this.ptFace0x[ixFacet + 1], this.ptFace0y[ixFacet + 1]);
					this.papplet.vertex(K.canvas_x0 + this.ptFace0x[ixFacet], this.ptFace0y[ixFacet]);
					this.papplet.vertex(K.canvas_x0 + this.ptFace1x[ixFacet], this.ptFace1y[ixFacet]);
					this.papplet.vertex(K.canvas_x0 + this.ptFace1x[ixFacet + 1], this.ptFace1y[ixFacet + 1]);
					this.papplet.endShape(PConstants.CLOSE);
				}
			}

			// ----- paint facet on file
			if ((Main.isAPPLIC) && (K.saveFileState == K.performingSAVEFILE)) {

				this.ps.setrgb((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

				this.ps.moveto(this.psFace0x[ixFacet + 1], this.psFace0y[ixFacet + 1]);
				this.ps.lineto(this.psFace0x[ixFacet], this.psFace0y[ixFacet]);
				this.ps.lineto(this.psFace1x[ixFacet], this.psFace1y[ixFacet]);
				this.ps.lineto(this.psFace1x[ixFacet + 1], this.psFace1y[ixFacet + 1]);
				this.ps.fillclosepath();
			}
		}
	}
}
