// ===================================================
// hairSynth - a simple vector graphics hair generator
// Copyright (C) 2007 JM Bara <http://www.codemuse.net>
// See licence terms in file "READ ME"
// ===================================================
// Uses Processing <http://processing.org>
// Uses SimplePostscript v.0092 <http:www.unlekker.net>
// ===================================================

/*****************************************************
 * class BufferBunch includes the methods that create
 * offsprings of the master hair strand, as well as
 * those which draw the canvas used on Step 2
 * (setting color, background, and zoom reticle)
 *
 * In order to accomplish a natural distribution of the
 * variations in color and shape among the offsprings of
 * the master hair strand, I used a recursive binary tree
 * approach for generating "offsprings" the the master
 * hair stand, and seeded random walk changes, as the tree
 * is being built.
 */


package hairSynth;

import processing.core.PApplet;

final class BufferBunch extends Buffer {

	// ----- No public vars here

	private long paramCodePopulated = -1; // not yet populated
	private long lastParamCodeDrawn = K.oscParamCodeLong;

	private int ixColorPopulated = -1; // not yet populated
	private int ixLastColorDrawn = KColor.ixColrButOn;
	private int ixLastBckgrDrawn = KColor.ixBckgButOn;

	private boolean isCopyToScreenSinceLastPoll = true;

	// ----- bounding box update
	private float tmpBoxLft = 10.0f; // set to impossible coordnates
	private float tmpBoxRgt = -10.0f;
	private float tmpBoxBot = 10.0f;
	private float tmpBoxTop = -10.0f;

	// ---- zoom calculations
	private int zoomCount = 0; // not yet set
	private float zoomXaccum;
	private float zoomYaccum;

	private Strand grandParent; // never created ... just pointer

	private Strand[] strand = new Strand[Buffer.numberOfStrandsTRAVERSED]; // absolute Max levels is 10

	private int ixCurrChild;

	private int firstThisBunch;
	private int firstNextBunch;
	private static int strandsBetweenRedraws = 16; // 32 results in about 10 FPS in my computer

	BufferBunch(final PApplet papplet_, final Strand parent_) {

		createPG(papplet_);
		this.grandParent = parent_; // not new(); shares strand from BufferParent
		for (int ixChild = 0; ixChild < Buffer.numberOfStrandsTRAVERSED; ixChild++) {
			this.strand[ixChild] = new Strand(ixChild);
		}
	}

	Strand getStrand(final int ixStrand) {
		return this.strand[ixStrand];
	}

	float getZoomAvgYpix() {

		if (this.zoomCount > 0) {
			return K.canvasCenter(this.zoomXaccum / this.zoomCount);
		}
		return K.canvasCenter(0.5f);
	}

	float getZoomAvgXpix() {

		if (this.zoomCount > 0) {
			return K.canvasBunchYpix(this.zoomYaccum / this.zoomCount);
		}
		return K.canvasBunchYpix(0.5f);
	}

	boolean getAndResetIsCopyToScreenSinceLastPoll() {

		boolean temp = this.isCopyToScreenSinceLastPoll;
		this.isCopyToScreenSinceLastPoll = false;
		return temp;
	}

	boolean getIsParamsOrColorsNeedPopulating() {
		return ((this.paramCodePopulated != K.oscParamCodeLong) || (this.ixColorPopulated != KColor.ixColrButOn));
	}

	boolean getIsParamsAndColorsArePopulated() {
		return ((this.paramCodePopulated == K.oscParamCodeLong) && (this.ixColorPopulated == KColor.ixColrButOn));
	}

	void reCenterAndTriggerReInitDraw() {

		if (this.stage > stageINITDRAW1) {
			K.setBoxBunchTmpXYandScaleIfLargerBoudariesThanSingle(this.tmpBoxLft, this.tmpBoxRgt, this.tmpBoxBot,
					this.tmpBoxTop);
			this.stage = stageINITDRAW1;
		}
	}

	void initTmpBoundaries() {
		this.tmpBoxLft = 10.0f; // set to impossible coordnates
		this.tmpBoxRgt = -10.0f;
		this.tmpBoxBot = 10.0f;
		this.tmpBoxTop = -10.0f;
	}

	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------
	// -------------------------------------------------------------------------

	void populateParamsOrColors() {

		if (this.paramCodePopulated != K.oscParamCodeLong) {

			this.grandParent.setPositionDxDyInit();
			this.strand[0].copyParamsFrom(this.grandParent);

			this.ixCurrChild = 0; // incremented before using
			makeTreeAndPopulateParams(Buffer.maxLEVEL, 0); // recursive
			this.paramCodePopulated = K.oscParamCodeLong;

		} else if (this.ixColorPopulated != KColor.ixColrButOn) {

			this.strand[0].clearDeltaHSBandCat5color();

			this.ixCurrChild = 0; // incremented before using
			makeTreeAndPopulateDeltaHSB(Buffer.maxLEVEL, 0); // recursive
			this.ixColorPopulated = KColor.ixColrButOn;

		}
	}

	boolean populateAndRender(final boolean isWorkedThisCycle) {

		boolean isWorkedThisMethod = true;

		if (this.getIsParamsOrColorsNeedPopulating()) {
			this.stage = stagePOPULATE0;
		} else if ((this.lastParamCodeDrawn != K.oscParamCodeLong) || (this.ixLastColorDrawn != KColor.ixColrButOn)
				|| (this.ixLastBckgrDrawn != KColor.ixBckgButOn)) {
			// ----- changed color scheme or background
			if (this.stage > stageINITDRAW1) {
				this.stage = stageINITDRAW1; // redraw
			}
		}

		switch (this.stage) {

		case stagePOPULATE0:

			this.populateParamsOrColors();

			if (this.getIsParamsAndColorsArePopulated()) {

				this.stage = stageINITDRAW1;
			}

			break;

		case stageINITDRAW1:

			this.clearPgScreen();
			this.copyToScreen();
			this.isCopyToScreenSinceLastPoll = true;

			this.lastParamCodeDrawn = K.oscParamCodeLong;
			this.ixLastColorDrawn = KColor.ixColrButOn;
			this.ixLastBckgrDrawn = KColor.ixBckgButOn;

			this.initTmpBoundaries();
			this.clearZoomAccum();
			this.firstNextBunch = 0;
			this.stage = stageDRAWABUNCH2;

			break;

		case stageDRAWABUNCH2: // Several Passes

			this.firstThisBunch = this.firstNextBunch;
			this.firstNextBunch += strandsBetweenRedraws;

			this.ixCurrChild = 0; // incremented before using
			drawSectionOfTreeAndReturnIsDone(Buffer.maxLEVEL, 0);

				this.copyToScreen();
				this.isCopyToScreenSinceLastPoll = true;

			if (this.firstNextBunch > Buffer.numberOfStrandsTRAVERSED) {
				this.stage = stageDONE4;
			}
			break;

		case stageREFRESHSCREEN3: // used when done but changed button choice

				this.copyToScreen();
				this.isCopyToScreenSinceLastPoll = true;

			this.stage = stageDONE4;
			break;

		case stageDONE4:
			K.currentChildBeingDrawn = 0;
			isWorkedThisMethod = false;
			break;
		default:
			isWorkedThisMethod = false;
			break;
		}
		return (isWorkedThisCycle || isWorkedThisMethod);
	}

	// =========================================================================
	// ALL PRIVATE BELOW
	// =========================================================================

	private void makeTreeAndPopulateParams(final int level_, final int idParent_) {

		if (level_ > 0) {

			double jiggleFactor = (level_ * level_) / Buffer.maxLEVEL2db; // more jiggle near trunk

			this.ixCurrChild++;
			this.strand[this.ixCurrChild].copyAndJiggleParams(this.strand[idParent_], jiggleFactor);
			makeTreeAndPopulateParams(level_ - 1, this.ixCurrChild);

			this.ixCurrChild++;
			this.strand[this.ixCurrChild].copyAndJiggleParams(this.strand[idParent_], jiggleFactor);
			makeTreeAndPopulateParams(level_ - 1, this.ixCurrChild);
		}
	}

	private void makeTreeAndPopulateDeltaHSB(final int level_, final int idParent_) {

		if (level_ > 0) {

			float shiftMin005toPlus005 = ((0.1f * idParent_) / Buffer.numberOfStrandsTRAVERSED) - 0.05f;

			float jiggleFactor = (level_ * level_) / Buffer.maxLEVEL2fl; // more jiggle near trunk

			this.ixCurrChild++;
			populateDeltaHSBandCat5color(idParent_, jiggleFactor, shiftMin005toPlus005);
			makeTreeAndPopulateDeltaHSB(level_ - 1, this.ixCurrChild);

			this.ixCurrChild++;
			populateDeltaHSBandCat5color(idParent_, jiggleFactor, shiftMin005toPlus005);
			makeTreeAndPopulateDeltaHSB(level_ - 1, this.ixCurrChild);
		}
	}

	private void populateDeltaHSBandCat5color(final int idParent_, final float jiggleFactor, final float shift) {

		float deltaHue = this.strand[idParent_].getDeltaHue() + K.randomInRangeFl(0.03f * jiggleFactor);
		float deltaSat = K.randomInRangeFl(0.3f) - shift;
		this.strand[this.ixCurrChild].setDeltaHSB(deltaHue, deltaSat, -deltaSat);
		this.strand[this.ixCurrChild].setCat5color(K.r.nextInt(KColor.totalCAT5));
	}

	private void drawSectionOfTreeAndReturnIsDone(final int level_, final int idParent_) {

		if (level_ > 0) {

			this.ixCurrChild++;
			drawSectionOfTreeAndReturnIsDone(level_ - 1, this.ixCurrChild);

			this.ixCurrChild++;
			drawSectionOfTreeAndReturnIsDone(level_ - 1, this.ixCurrChild);

		} else if ((idParent_ >= this.firstThisBunch) && (idParent_ < this.firstNextBunch)) { // Level = 0
			drawStrand(idParent_);
		}
	}

	private void clearZoomAccum() {

		this.zoomXaccum = 0.0f;
		this.zoomYaccum = 0.0f;
		this.zoomCount = 0;
	}

	private void drawStrand(final int idParent_) {

		K.currentChildBeingDrawn = idParent_;
		Strand sParent_ = this.strand[idParent_];

		this.pg.beginDraw();

		// ----- calculation of stroke colors

		int rgb = KColor.getMedColor(sParent_);
		this.pg.stroke((rgb >> 16) & 0xFF, (rgb >> 8) & 0xFF, rgb & 0xFF);

		double dist = 0.0;

		boolean zoomNotYetCaptured = true;

		float face1Xinit = (float) sParent_.getFace1DXInit();
		float face1Yinit = (float) sParent_.getFace1DYInit();

		if (this.tmpBoxLft > face1Xinit) {
			this.tmpBoxLft = face1Xinit;
		}

		if (this.tmpBoxRgt < face1Xinit) {
			this.tmpBoxRgt = face1Xinit;
		}

		if (this.tmpBoxBot > face1Yinit) {
			this.tmpBoxBot = face1Yinit;
		}

		if (this.tmpBoxTop < face1Yinit) {
			this.tmpBoxTop = face1Yinit;
		}

		while (dist < K.totalStrandLEN) {

			dist = sParent_.nextSegment(dist);

			float face1X = sParent_.getFace1Xfloat();
			float face1Y = sParent_.getFace1Yfloat();

			if (this.tmpBoxLft > face1X) {
				this.tmpBoxLft = face1X;
			}

			if (this.tmpBoxRgt < face1X) {
				this.tmpBoxRgt = face1X;
			}

			if (this.tmpBoxBot > face1Y) {
				this.tmpBoxBot = face1Y;
			}

			if (this.tmpBoxTop < face1Y) {
				this.tmpBoxTop = face1Y;
			}

			this.pg.line(K.canvasCenter(sParent_.getFace0Xfloat()), K.canvasBunchYpix(sParent_.getFace0Yfloat()), K
					.canvasCenter(face1X), K.canvasBunchYpix(face1Y));

			if (zoomNotYetCaptured && (dist > K.distZOOM)) {

				this.zoomXaccum += sParent_.getFace1Xfloat();
				this.zoomYaccum += sParent_.getFace1Yfloat();
				this.zoomCount++;
				zoomNotYetCaptured = false;
			}
		}
		this.pg.endDraw();
	}

}
