/* this class holds the methods that calculate the forces and displacements
 * of each grid node.
 */

// Note: replaced "bending" ( force proportional to distance (displacement) between
// current point and mid-point of line connecting "before" and "after"
// (similar to elastic force, but rest = zero)) ...
//
// ... with, "stay straight" force emanating from each side (mooring), linearly
// reduced as it travels away from moorings (as if cloth threads become thinner)

public final class Calc {

	// Grid Rip Vertical (meaning Y changes) coordinates
	// Note: Rips must be at least one cell away from border! ...
	// ... because "virtual" nodes are extrapolated

	public static GridNodeC[][] node;


	//XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX
	private Calc() {
		/* empty */
	}
	//XXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXXX

	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////

	static float getNodeX(final int i, final int j) {

		if ((i < 0) || (i > K.gridSize) || (j < 0) || (j > K.gridSize)) {
			return i;
		}
		return node[i][j].x;
	}

	static float getNodeY(final int i, final int j) {

		if ((i < 0) || (i > K.gridSize) || (j < 0) || (j > K.gridSize)) {
			return j;
		}
		return node[i][j].y;
	}

	static float getManhDist(final int i0, final int j0, final int i1, final int j1) {

		return (Math.abs(getNodeX(i1, j1) - getNodeX(i0, j0)) + Math.abs(getNodeY(i1, j1) - getNodeY(i0, j0)));

	}

	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////

	public static void init() {

		createGrid();
		resetGridXY();
		mendRips();
	}

	private static void createGrid() {

		node = new GridNodeC[K.gridSizeMAX + 1][K.gridSizeMAX + 1];

		for (int i = 0; i <= K.gridSizeMAX; i++) {
			for (int j = 0; j <= K.gridSizeMAX; j++) {
				node[i][j] = new GridNodeC();
			}
		}
	}

	public static void resetGridXY() {

		for (int i = 0; i <= K.gridSize; i++) {
			for (int j = 0; j <= K.gridSize; j++) {

				// -----locate position in grid
				node[i][j].x = i;
				node[i][j].y = j;

				// -----pin border rows to frame
				if ((i == 0) || (i == K.gridSize) || (j == 0) || (j == K.gridSize)) {
					node[i][j].isNodeFree = false;
				} else {
					node[i][j].isNodeFree = true;
				}
			}
		}
	}

	public static void mendRips() {
		for (int i = 0; i <= K.gridSize; i++) {
			for (int j = 0; j <= K.gridSize; j++) {

				// -----make all springs active (deactivated with rips)
				node[i][j].isSpringRightActive = true;
				node[i][j].isSpringDownActive = true;
			}
		}
	}

	public static void createDiagonalRip(final float ripX0, final float ripY0, final float ripX1, final float ripY1) {

		// determine line slant ... Note horizontal line creates div by zero
		boolean isVertical = (ripX1 == ripX0);
		float ripSlant = 0.0f;
		if (!isVertical) {
			ripSlant = (ripY1 - ripY0) / (ripX1 - ripX0);
		}

		// -----determine first and last row ripped--------
		int ripTopRow = (int) Math.min(ripY0, ripY1) + 1; // ceiling
		int ripBotRow = (int) Math.max(ripY0, ripY1); // floor

		for (int j = ripTopRow; j <= ripBotRow; j++) {
			float ripXexact = ripX0;
			if (!isVertical) {
				ripXexact += (j - ripY0) / ripSlant;
			}
			int ripLeftX = (int) ripXexact;
			node[ripLeftX][j].isSpringRightActive = false;
			node[ripLeftX][j].springRightStub = ripXexact - ripLeftX;
		}

		// -----determine first and last column ripped-----
		int ripLeftCol = (int) Math.min(ripX0, ripX1) + 1; // ceiling
		int ripRightCol = (int) Math.max(ripX0, ripX1); // floor

		for (int i = ripLeftCol; i <= ripRightCol; i++) {
			float ripYexact = ripY0 + ((i - ripX0) * ripSlant);
			int ripTopY = (int) ripYexact;
			node[i][ripTopY].isSpringDownActive = false;
			node[i][ripTopY].springDownStub = ripYexact - ripTopY;
		}
	}

	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	// ////////////////////////////////////////////////////////////////////////
	public static void deformGrid(final boolean isSetBoundaries) {

		calculateSpringTension();

		if (K.bending2Elastic > 0.0) {
			calculateBendingForces();
		}
		calculateNodePositions(isSetBoundaries);
	}

	private static void calculateSpringTension() { // also records spring lengths

		// uses (double) beacause of Math.sqrt
		// Note: skips right-most column and bottom-most row
		for (int i = 0; i < K.gridSize; i++) {
			for (int j = 0; j < K.gridSize; j++) {
				// Calculate springRightForce
				if (node[i][j].isSpringRightActive) {
					// Note: all coordinates are expressed in CellWdith and CellHeight units
					float dx = node[i + 1][j].x - node[i][j].x;
					float dy = node[i + 1][j].y - node[i][j].y;
					float springLength = (float) Math.sqrt((dx * dx) + (dy * dy));
					float force = springLength - K.springSizAtRest;

					// float force = springLength * springLength ;
					if (springLength > 0) {
						node[i][j].springRightForceX = force * (dx / springLength);
						node[i][j].springRightForceY = force * (dy / springLength);
					} else {
						node[i][j].springRightForceX = 0.0f;
						node[i][j].springRightForceY = 0.0f;
					}

					node[i][j].springRightLength = springLength; // used for bending, below
				}

				// Calculate springDownForce
				if (node[i][j].isSpringDownActive) {
					// Note: all coordinates are expressed in CellWdith and CellHeight units
					float dx = node[i][j + 1].x - node[i][j].x;
					float dy = node[i][j + 1].y - node[i][j].y;
					float springLength = (float) Math.sqrt((dx * dx) + (dy * dy));
					float force = springLength - K.springSizAtRest;

					// float force = springLength * springLength ;
					if (springLength > 0) {
						node[i][j].springDownForceX = force * (dx / springLength);
						node[i][j].springDownForceY = force * (dy / springLength);
					} else {
						node[i][j].springDownForceX = 0.0f;
						node[i][j].springDownForceY = 0.0f;
					}

					node[i][j].springDownLength = springLength; // used for bending, below
				}
			}
		}
	}

	private static void calculateBendingForces() {
		// Note: distances calculated in CellWidth units
		for (int i = 0; i <= K.gridSize; i++) {
			for (int j = 0; j <= K.gridSize; j++) {
				// needs to be done in separate previous loop as next loop...
				// ...acesses cells to the left and to the right
				node[i][j].accumBendingForceX = 0.0f;
				node[i][j].accumBendingForceY = 0.0f;
			}
		}

		for (int j = 1; j <= (K.gridSize - 1); j++) {
			// calculate bending force at right of angle
			int i = 1;

			while ((i < (K.gridSize - 1)) && node[i][j].isSpringRightActive) {
				calculateNewBendingForce(i - 1, j, i, j, i + 1, j); // far.. close .. target (where bending
				// force is)
				i++;
			}

			// calculate bending force at left of angle
			i = K.gridSize - 1;

			while ((i > 1) && (node[i - 1][j].isSpringRightActive)) {
				calculateNewBendingForce(i + 1, j, i, j, i - 1, j); // far.. close .. target (where bending
				// force is)
				i--;
			}
		}

		// scan vertically
		for (int i = 1; i <= (K.gridSize - 1); i++) {
			// calculate bending force below of angle
			int j = 1;

			while ((j < (K.gridSize - 1)) && node[i][j].isSpringDownActive) {
				calculateNewBendingForce(i, j - 1, i, j, i, j + 1); // far.. close .. target (where bending
				// force is)
				j++;
			}

			// calculate bending force above angle
			j = K.gridSize - 1;

			while ((j > 1) && node[i][j - 1].isSpringDownActive) {
				calculateNewBendingForce(i, j + 1, i, j, i, j - 1);
				j--;
			}
		}
	}

	private static void calculateNewBendingForce(final int ai, final int aj, final int bi, final int bj, final int ci,
			final int cj) {
		// prolong a-->b by dist b-->c
		float distAB;
		float distBC;

		if (bj == aj) { // horizontal

			if (bi > ai) { // left to right
				distAB = node[ai][aj].springRightLength;
				distBC = node[bi][bj].springRightLength;
			} else { // right to left
				distAB = node[bi][bj].springRightLength;
				distBC = node[ci][cj].springRightLength;
			}
		} else { // vertical

			if (bj > aj) { // top to down
				distAB = node[ai][aj].springDownLength;
				distBC = node[bi][bj].springDownLength;
			} else { // bottom to up
				distAB = node[bi][bj].springDownLength;
				distBC = node[ci][cj].springDownLength;
			}
		}

		if (distAB <= 0.0) {
			distAB = 1.0f; // arbitrary... only here to avoid div by zero
		}

		float extendABx = node[bi][bj].x + ((distBC * (node[bi][bj].x - node[ai][aj].x)) / distAB);
		float extendABy = node[bi][bj].y + ((distBC * (node[bi][bj].y - node[ai][aj].y)) / distAB);

		float bendingForceX = extendABx - node[ci][cj].x; // * bendingConstant applied later;
		float bendingForceY = extendABy - node[ci][cj].y; // * bendingConstant applied later;

		node[ci][cj].accumBendingForceX += bendingForceX;
		node[ci][cj].accumBendingForceY += bendingForceY;
	}

	private static void calculateNodePositions(final boolean isSetBoundaries) {

		if (isSetBoundaries) {
			K.xMinCells = 0;
			K.xMaxCells = K.gridSize;
			K.yMinCells = 0;
			K.yMaxCells = K.gridSize;
		}

		// Note: displacement calculated in CellWidth units
		for (int i = 0; i <= K.gridSize; i++) {
			for (int j = 0; j <= K.gridSize; j++) {
				if (node[i][j].isNodeFree) {
					float dispX = 0.0f;
					float dispY = 0.0f;

					// spring Right
					if (node[i][j].isSpringRightActive) {
						dispX += node[i][j].springRightForceX;
						dispY += node[i][j].springRightForceY;
					}

					// spring Down
					if (node[i][j].isSpringDownActive) {
						dispX += node[i][j].springDownForceX;
						dispY += node[i][j].springDownForceY;
					}

					// spring to the Left
					if (node[i - 1][j].isSpringRightActive) {
						dispX -= node[i - 1][j].springRightForceX;
						dispY -= node[i - 1][j].springRightForceY;
					}

					// spring to the Top
					if (node[i][j - 1].isSpringDownActive) {
						dispX -= node[i][j - 1].springDownForceX;
						dispY -= node[i][j - 1].springDownForceY;
					}

					dispX += (node[i][j].accumBendingForceX * K.bending2Elastic);
					dispY += (node[i][j].accumBendingForceY * K.bending2Elastic);

					node[i][j].x += dispX * K.elasticConstant;
					node[i][j].y += dispY * K.elasticConstant;

					if (isSetBoundaries) {
						float x_ = node[i][j].x;
						float y_ = node[i][j].y;
						if (K.xMinCells > x_) {
							K.xMinCells = x_;
						} else if (K.xMaxCells < x_) {
							K.xMaxCells = x_;
						}
						if (K.yMinCells > y_) {
							K.yMinCells = y_;
						} else if (K.yMaxCells < y_) {
							K.yMaxCells = y_;
						}
					}
				}
			}
		}
	}
}
